﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using System.Xml.Serialization;
using UnityEngine;

public class SettingsManager : MonoBehaviour
{
    public Dictionary<Constants.Settings, object> Settings { get; private set; }
    private static Dictionary<Constants.Settings, object> DefaultSettings = new Dictionary<Constants.Settings, object>();

    private static SettingsManager _Instance;
    public static SettingsManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<SettingsManager>();
            }

            return _Instance;
        }
    }

    private void Awake()
    {
        GetDefaultSettings();   //Fill the DefaultSettings dictionary

        if (File.Exists(Constants.SettingsXMLFilePath))
        {
            Settings = ReadSettingsFile();  //File exists, let's read the settings

            if(Settings == null)    //Null was returned so something went wrong, let's delete the settings file and start again with the defaults
            {
                File.Delete(Constants.SettingsXMLFilePath);
                CreateNewSettingsFile();
            }
        }

        else
        {
            CreateNewSettingsFile();    //No file exists, let's create one with the defaults
        }
    }

    private Dictionary<Constants.Settings, object> ReadSettingsFile()
    {
        try
        {
            if (VerifyAllSettingsPresent()) //All settings have entries in the file
            {
                Dictionary<Constants.Settings, object> retDict = new Dictionary<Constants.Settings, object>();
                XDocument settingsXML = XDocument.Load(Constants.SettingsXMLFilePath);

                foreach (var setting in settingsXML.Descendants("Setting"))
                {
                    //Add the name of the setting parsed into an enum and its value into the dictionary
                    retDict.Add((Constants.Settings)Enum.Parse(typeof(Constants.Settings), setting.Attribute("Name").Value.ToString(), true), setting.Value);
                }

                return retDict;
            }

            else
            {
                return null;    //We're missing settings so something's wrong, return null
            }
        }

        catch(Exception ex)
        {
            Debug.Log("Caught an exception when reading settings file, returning false and creating new file. The exception is: " + ex);
            return null;
        }
    }

    private bool VerifyAllSettingsPresent()
    {
        XDocument settingsXML = XDocument.Load(Constants.SettingsXMLFilePath);

        var settingsXMLList = (from ele in settingsXML.Descendants("Setting")
                               select ele).ToList();    //Get all the setting elements

        foreach (string enumVal in Enum.GetNames(typeof(Constants.Settings)))
        {
            //Loop through all values in the enum
            bool foundEnumValue = false;

            foreach (XElement settingElem in settingsXMLList)
            {
                //Loop through all the settings
                if (settingElem.Attribute("Name").Value == enumVal)
                {
                    //This setting has an entry, found it, so break from the settings loop and move onto the next enum entry
                    foundEnumValue = true;
                    break;
                }
            }

            if (!foundEnumValue)
            {
                //We didn't find it so we have missing settings, return false
                return false;
            }

            foundEnumValue = false;
        }

        //We didn't return in the loop so we've got all settings, return true
        return true;
    }

    private static void GetDefaultSettings()
    {
        //Create the default settings dictionary
        DefaultSettings = new Dictionary<Constants.Settings, object>();
        DefaultSettings.Add(Constants.Settings.MusicVolume, 0.5f);
        DefaultSettings.Add(Constants.Settings.SFXVolume, 1.0f);
    }

    private void CreateNewSettingsFile()
    {
        File.Delete(Constants.SettingsXMLFilePath); //Delete one if it exists

        XDocument settingsXML = new XDocument(
                                            new XDeclaration("1.0", "utf-8", null),
                                            new XElement("Root",
                                            from ent in DefaultSettings
                                            select new XElement("Setting", new XAttribute("Name", ent.Key.ToString()), ent.Value)
                                            ));

        //Create a new document by looping through each default settings entry

        settingsXML.Save(Constants.SettingsXMLFilePath);


        //The file has been made and saved, let's make the Settings dictionary with the defaults
        Settings = new Dictionary<Constants.Settings, object>();
        foreach(var defaultSetting in DefaultSettings)
        {
            Settings[defaultSetting.Key] = defaultSetting.Value;
        }
    }

    /// <summary>
    /// Writes the settings dictionary to the Settings XML file
    /// </summary>
    public void WriteSettings()
    {
        XDocument settingsXML = new XDocument(
                                        new XDeclaration("1.0", "utf-8", "yes"),
                                        new XElement("Root", Settings.Select(val => new XElement("Setting", new XAttribute("Name", val.Key), val.Value))
                                            ));

        //Create a new document by looping through each default settings entry then save it

        settingsXML.Save(Constants.SettingsXMLFilePath);
    }
}
